<?php
/* --------------------------------------------------------------
   ThemeContentsParser.inc.php 2019-06-21
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * Class ThemeContentsParser
 *
 * This class represents a parser for theme contents
 */
class ThemeContentsParser
{
    protected static $identificationFactory;
    
    
    /**
     * Return the parsed theme contents
     *
     * @param stdClass $contents Theme contents
     *
     * @return ThemeContents
     * @throws UnfinishedBuildException
     */
    public static function parse(stdClass $contents)
    {
        $properties = ['infoPages', 'linkPages', 'infoElements'];
        
        //  Ensuring that if a property is not send or needed the collection can be constructed
        foreach ($properties as $property) {
            if (!$contents->$property) {
                $contents->$property = [];
            }
        }
        
        $infoPageContents    = self::parseInfoPageContents($contents->infoPages);
        $linkPageContents    = self::parseLinkPageContents($contents->linkPages);
        $infoElementContents = self::parseInfoElementContents($contents->infoElements);
        
        return new ThemeContents($infoPageContents, $linkPageContents, $infoElementContents);
    }
    
    
    /**
     * Return the parsed info page contents
     *
     * @param array $infoPageContents Info page contents
     *
     * @return InfoPageContentCollection
     * @throws UnfinishedBuildException
     */
    private static function parseInfoPageContents(array $infoPageContents)
    {
        $contents = [];
        
        foreach ($infoPageContents as $infoPageContent) {
            $contents[] = self::parseInfoPageContent($infoPageContent);
        }
        
        return new InfoPageContentCollection($contents);
    }
    
    
    /**
     * Return the parsed link page contents
     *
     * @param array $linkPageContents Link page contents
     *
     * @return LinkPageContentCollection
     * @throws UnfinishedBuildException
     */
    private static function parseLinkPageContents(array $linkPageContents)
    {
        $contents = [];
        
        foreach ($linkPageContents as $linkPageContent) {
            $contents[] = self::parseLinkPageContent($linkPageContent);
        }
        
        return new LinkPageContentCollection($contents);
    }
    
    
    /**
     * Return the parsed info element contents
     *
     * @param array $infoElementContents Info element contents
     *
     * @return InfoElementContentCollection
     * @throws UnfinishedBuildException
     */
    private static function parseInfoElementContents(array $infoElementContents)
    {
        $contents = [];
        
        foreach ($infoElementContents as $infoElementContent) {
            $contents[] = self::parseInfoElementContent($infoElementContent);
        }
        
        return new InfoElementContentCollection($contents);
    }
    
    
    /**
     * Return a parsed info page content
     *
     * @param stdClass $content Info page content
     *
     * @return InfoPageContent
     */
    private static function parseInfoPageContent(stdClass $content): \InfoPageContent
    {
        if ($content->type !== ContentType::CONTENT) {
            throw new ParseError('Content type does not match');
        }
        
        /**/
        $position     = PagePosition::createFromString($content->position);
        $contentGroup = $content->id ?? null;
        $contentAlias = $content->alias ?? null;
        $contentId    = self::identificationFactory()->forPreference($contentAlias, $contentGroup);
        
        $status          = MainFactory::create(ContentStatus::class, (bool)$content->status);
        $names           = self::createLocalizedCollection($content->name, new StringType(ContentName::class));
        $titles          = self::createLocalizedCollection($content->title, new StringType(ContentTitle::class));
        $headings        = self::createLocalizedCollection($content->heading, new StringType(ContentHeading::class));
        $texts           = self::createLocalizedCollection($content->text, new StringType(ContentText::class));
        $downloads       = self::createLocalizedCollection($content->downloadFile,
                                                           new StringType(ContentDownloadFile::class));
        $metaTitles      = self::createLocalizedCollection($content->metaTitle,
                                                           new StringType(ContentMetaTitle::class));
        $metaKeywords    = self::createLocalizedCollection($content->metaKeywords,
                                                           new StringType(ContentMetaKeywords::class));
        $metaDescription = self::createLocalizedCollection($content->metaDescription,
                                                           new StringType(ContentMetaDescription::class));
        $urlKeywords     = self::createLocalizedCollection($content->urlKeywords,
                                                           new StringType(ContentUrlKeywords::class));
        $urlRewrites     = self::createLocalizedCollection($content->urlRewrite,
                                                           new StringType(ContentUrlRewrite::class));
        
        $sitemap = null;
        if (null !== $content->sitemap) {
            $sitemapVisibility = MainFactory::create(ContentSitemapVisibility::class,
                                                     $content->sitemap->visible ? : false);
            $sitemapPriorities = self::createLocalizedCollection($content->sitemap->priority,
                                                                 new StringType(ContentSitemapPriority::class));
            $sitemapFrequency  = ContentSitemapChangeFrequency::createFromString($content->sitemap->changeFrequency);
            $sitemap           = MainFactory::create(ContentSitemap::class,
                                                     $sitemapVisibility,
                                                     $sitemapPriorities,
                                                     $sitemapFrequency);
        }
        
        $allowRobotsStatus = MainFactory::create(ContentAllowRobotsStatus::class, $content->allowRobots ? : false);
        $opengraphImages   = self::createLocalizedCollection($content->opengraphImage,
                                                             new StringType(ContentOpengraphImage::class));
        
        return ContentFactory::newInfoPageContent()
            ->inPosition($position)
            ->usingId($contentId)
            ->usingStatus($status)
            ->usingNames($names)
            ->usingTitles($titles)
            ->usingHeadings($headings)
            ->usingTexts($texts)
            ->usingDownloadFiles($downloads)
            ->usingMetaTitles($metaTitles)
            ->usingMetaKeywords($metaKeywords)
            ->usingMetaDescriptions($metaDescription)
            ->usingUrlKeywords($urlKeywords)
            ->usingUrlRewrites($urlRewrites)
            ->usingSitemap($sitemap)
            ->usingAllowRobotsStatus($allowRobotsStatus)
            ->usingOpengraphImages($opengraphImages)
            ->build();
    }
    
    
    /**
     * Return a parsed link page content
     *
     * @param stdClass $content Link page content
     *
     * @return LinkPageContent
     * @throws UnfinishedBuildException
     */
    private static function parseLinkPageContent(stdClass $content)
    {
        if ($content->type !== ContentType::LINK) {
            throw new ParseError('Content type does not match');
        }
        
        $position = PagePosition::createFromString($content->position);
        
        $status             = MainFactory::create(ContentStatus::class, (bool)$content->status);
        $names              = self::createLocalizedCollection($content->name, new StringType(ContentName::class));
        $titles             = self::createLocalizedCollection($content->title, new StringType(ContentTitle::class));
        $links              = self::createLocalizedCollection($content->link, new StringType(ContentLink::class));
        $contentGroup       = $content->id ?? null;
        $contentAlias       = $content->alias ?? null;
        $contentId          = self::identificationFactory()->forPreference($contentAlias, $contentGroup);
        $openInNewTabStatus = MainFactory::create(ContentOpenInNewTabStatus::class, (bool)$content->openInNewTab);
        
        return ContentFactory::newLinkPageContent()
            ->inPosition($position)
            ->usingId($contentId)
            ->usingStatus($status)
            ->usingNames($names)
            ->usingTitles($titles)
            ->usingLinks($links)
            ->usingOpenInNewTabStatus($openInNewTabStatus)
            ->build();
    }
    
    
    /**
     * Return a parsed info element content
     *
     * @param stdClass $content Info element content
     *
     * @return InfoElementContent
     * @throws UnfinishedBuildException
     */
    private static function parseInfoElementContent(stdClass $content)
    {
        if ($content->type !== ContentType::CONTENT) {
            throw new ParseError('Content type does not match');
        }
        
        $position = ElementPosition::createFromString($content->position);
        $status   = MainFactory::create(ContentStatus::class, $content->status);
        $titles   = self::createLocalizedCollection($content->title, new StringType(ContentTitle::class));
        $headings = self::createLocalizedCollection($content->heading, new StringType(ContentHeading::class));
        $texts    = self::createLocalizedCollection($content->text, new StringType(ContentText::class));
        $contentGroup       = $content->id ?? null;
        $contentAlias       = $content->alias ?? null;
        $contentId          = self::identificationFactory()->forPreference($contentAlias, $contentGroup);
        return ContentFactory::newInfoElementContent()
            ->inPosition($position)
            ->usingId($contentId)
            ->usingStatus($status)
            ->usingTitles($titles)
            ->usingHeadings($headings)
            ->usingTexts($texts)
            ->build();
    }
    
    
    /**
     *
     */
    protected static function identificationFactory(): ContentIdentificationFactoryInterface
    {
        if (static::$identificationFactory === null) {
            static::$identificationFactory = MainFactory::create(ContentIdentificationFactory::class);
        }
        
        return static::$identificationFactory;
    }
    
    
    /**
     * Return a collection of the provided localized values
     *
     * @param stdClass    $values        Object containing locale-based values
     * @param \StringType $itemClassName Item class name
     *
     * @return mixed Dynamic output (all outputs implement LocalizedContentAttributeCollectionInterface)
     */
    public static function createLocalizedCollection(
        stdClass $values = null,
        StringType $itemClassName
    ) {
        if ($values === null) {
            return null;
        }
        
        $items               = [];
        $collectionClassName = $itemClassName->asString() . 'Collection';
        
        foreach (get_object_vars($values) as $languageCode => $name) {
            if ($itemClassName->asString() === 'ContentSitemapPriority') {
                $firstParameter = (float)($name);
            } else {
                $firstParameter = (string)$name;
            }
            
            $languageCodeObject = MainFactory::create('LanguageCode', new StringType($languageCode));
            $items[]            = MainFactory::create($itemClassName->asString(), $firstParameter, $languageCodeObject);
        }
        
        return MainFactory::create($collectionClassName, $items);
    }
}